<?php

namespace App\Http\Controllers\Backend\Addon;

use App\Http\Controllers\Controller;
use App\Models\Addon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\ValidationException;
use ZipArchive;

class AddonController extends Controller
{
    const PRODUCT_ID = '48918061';

    private function getAddonsFromServer()
    {
        $license_server = 'https://springsoftit.com/api/addons';

        try {
            $response = Http::timeout(60)->get($license_server);

            $addons = collect($response->json())->filter(function ($addon) {
                return isset($addon['mother_product_code']) && $addon['mother_product_code'] == self::PRODUCT_ID;
            });

            return $addons->values()->all();
        } catch (\Throwable $th) {
        }

        return [];
    }

    private function getAddonById($id)
    {
        $addons = $this->getAddonsFromServer();

        return collect($addons)->where('id', $id)->first();
    }

    public function addon()
    {
        $addons = $this->getAddonsFromServer();

        return view('backend.addon.index', compact('addons'));
    }

    public function installAddon($id)
    {
        $addon = $this->getAddonById($id);

        if ($addon['status'] == 3) {
            return redirect()->route('panel.addons')->with('error', 'Addon is not published yet please wait for publishing');
        }

        return view('backend.addon.install', compact('addon'));
    }

    function changeStatus($id)
    {
        $addon = Addon::findOrFail($id);

        $addon->status = !$addon->status;
        $addon->save();

        $this->updateModuleStatusesJson($addon->name, $addon->status);

        $seederClass = 'Modules\\' . ucfirst($addon->name) . '\\App\\Helper';

        if (class_exists($seederClass)) {
            $seederClass::neccessaryUpdate($addon);
        }

        return redirect()->route('panel.addons')->with('success', 'Addon status changed successfully');
    }

    public function installAddonStore(Request $request, $id)
    {
        $request->validate([
            'licence_key' => 'required',
            'code_username' => 'required',
            'file' => 'required|mimes:zip',
        ]);

        $addon = $this->getAddonById($id);

        
        if ($addon['status'] == 3) {
            return redirect()->route('panel.addons')->with('error', 'Addon is not published yet please wait for publishing');
        }

        $response = $this->verifyPurchase($request->licence_key, $addon['product_code']);

        $responseFromRemote = $this->checkLicense($request->licence_key, $request->code_username, $request->getHost());
        if (!$responseFromRemote['status']) {
            return back()->with('error', $responseFromRemote['message']);
        }

        if (!$response['status']) {
            return back()->with('error', $response['message']);
        }

        if (Addon::where('name', $addon['addon_placeholder'])->exists()) {
            return redirect()->route('panel.addons')->with('error', 'Addon already installed');
        }

        $filename = $this->storeAddonFile($request->file('file'));

        $zipPath = cons()::IMAGE_UPLOAD_ROOT_PATH . 'addons/' . $filename;



        if (!$this->checkValidFile($filename, $addon['addon_placeholder'])) {

            unlink($zipPath);

            return redirect()->back()->with('error', 'Invalid Zip file uploaded.');
        }

        if (!$this->checkLicenseWithMotherProductCode($filename, $addon['addon_placeholder'], $addon['mother_product_code'])) {
            unlink($zipPath);
            return redirect()->back()->with('error', 'Invalid addon for this product.');
        }

        if ($this->extractAddonFile($filename)) {
            $migrationPath = base_path('Modules/' . $addon['addon_placeholder'] . '/Database/Migrations');
            if (File::exists($migrationPath)) {
                Artisan::call('migrate', ['--path' => 'Modules/' . $addon['addon_placeholder'] . '/Database/Migrations', '--force' => true]);
            }

            $addonCreated = $this->updateAddonStatus($addon, $request);

            $this->updateModuleStatusesJson(ucfirst($addon['addon_placeholder']), $addonCreated->status);

            $seederClass = 'Modules\\' . ucfirst($addon['addon_placeholder']) . '\\Database\\Seeders\\ModuleSeeder';

            if (class_exists($seederClass)) {
                Artisan::call('db:seed', ['--class' => $seederClass]);
            }

            $this->sendInstallationToLicenseServer($request);

            $seederClass = 'Modules\\' . ucfirst($addonCreated->name) . '\\App\\Helper';

            if (class_exists($seederClass)) {
                $seederClass::neccessaryUpdate($addonCreated);
            }

            return redirect()->route('panel.addons')->with('success', 'Addon installed successfully');
        }

        return redirect()->route('panel.addons')->with('error', 'Failed to extract the addon.');
    }

    function checkValidFile($zipPath, $addon)
    {
        $path = cons()::IMAGE_UPLOAD_ROOT_PATH . 'addons/' . $zipPath;

        if (!file_exists($path) || !is_readable($path)) {
            return false;
        }

        $zip = new ZipArchive();

        if ($zip->open($path) === true) {
            $index = $zip->locateName(ucwords($addon) . '/licence.txt');

            $zip->close();

            return $index !== false;
        }

        return false;
    }

    private function checkLicenseWithMotherProductCode($zipPath, $module, $mother)
    {
        $path = cons()::IMAGE_UPLOAD_ROOT_PATH . 'addons/' . $zipPath;

        $zip = new ZipArchive();
        if ($zip->open($path) === true) {
            $licenseContents = $zip->getFromName($module.'/licence.txt');


            if (trim($licenseContents) !== trim($mother)) {
                $zip->close();
                return false;
            }

            $zip->close();
        }

        return true;
    }

    function checkLicense($purchase_code, $user, $current_domain)
    {
        $license_server = 'https://springsoftit.com/api/installations';

        try {
            $response = Http::post($license_server, [
                'code' => $purchase_code,
                'user' => $user,
                'domain' => $current_domain,
                'ip' => request()->ip(),
            ]);

            return $response->json();
        } catch (\Throwable $th) {
            //throw $th;
        }

        return ['status' => false, 'message' => 'Could not connect to server'];
    }

    function verifyPurchase($purchase_code, $code)
    {
        $personalToken = 'PbldGXTOooIBQmOfAOn0VuvfASGe6C2n';

        $code = trim($purchase_code);

        $expectedProductId = $code;

        if (!preg_match("/^([a-f0-9]{8})-(([a-f0-9]{4})-){3}([a-f0-9]{12})$/i", $code)) {
            return ['status' => false, 'message' => 'Invalid purchase code'];
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => "https://api.envato.com/v3/market/author/sale?code={$code}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_HTTPHEADER => ["Authorization: Bearer {$personalToken}", 'User-Agent: Purchase code verification script'],
        ]);

        $response = @curl_exec($ch);
        $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch) > 0) {
            return ['status' => false, 'message' => 'Failed to connect: ' . curl_error($ch)];
        }

        switch ($responseCode) {
            case 404:
                return ['status' => false, 'message' => 'Invalid purchase code'];
            case 403:
                return ['status' => false, 'message' => 'The personal token is missing the required permission for this script'];
            case 401:
                return ['status' => false, 'message' => 'The personal token is invalid or has been deleted'];
        }

        if ($responseCode !== 200) {
            return ['status' => false, 'message' => "Got status {$responseCode}, try again shortly"];
        }

        $body = @json_decode($response);

        if ($body === false && json_last_error() !== JSON_ERROR_NONE) {
            return ['status' => false, 'message' => 'Error parsing response, try again'];
        }

        if (!isset($body->item->id) || $body->item->id != $expectedProductId) {
            return ['status' => false, 'message' => 'This purchase code is not valid for your product'];
        }

        return ['status' => true, 'message' => $body];
    }

    private function storeAddonFile($file)
    {
        $filename = storeFile($file, 'addons');
        return $filename;
    }

    private function extractAddonFile($filename)
    {
        $zipPath = cons()::IMAGE_UPLOAD_ROOT_PATH . 'addons/' . $filename;
        $extractPath = base_path('Modules');
        $zip = new ZipArchive();

        if ($zip->open($zipPath) === true) {
            $zip->extractTo($extractPath);
            $zip->close();
            unlink($zipPath);

            return true;
        }

        return false;
    }

    public function deleteModuleFolder($moduleName)
    {
        $modulePath = base_path("Modules/{$moduleName}");

        if (File::exists($modulePath) && File::isDirectory($modulePath)) {
            File::deleteDirectory($modulePath);

            return response()->json(['message' => 'Module folder deleted successfully'], 200);
        }

        return response()->json(['message' => 'Module folder not found'], 404);
    }

    private function updateAddonStatus($addon, $request)
    {
        $isAvailable = Addon::where('purchase_key', $request->licence_key)->first();

        if ($isAvailable) {
            $this->deleteModuleFolder($addon['addon_placeholder']);

            throw ValidationException::withMessages([
                'licence_key' => ['This license key is already in use.'],
            ]);
        }

        $isAvailable = Addon::create([
            'name' => $addon['addon_placeholder'],
            'purchase_key' => $request->licence_key,
            'code_username' => $request->code_username,
            'status' => 0,
        ]);

        return $isAvailable;
    }

    private function updateModuleStatusesJson($name, $status = true)
    {
        $jsonFilePath = base_path('modules_statuses.json');

        if (File::exists($jsonFilePath)) {
            $jsonData = json_decode(File::get($jsonFilePath), true);
        } else {
            $jsonData = [];
        }

        $jsonData[$name] = $status ? true : false;

        File::put($jsonFilePath, json_encode($jsonData, JSON_PRETTY_PRINT));
    }

    private function sendInstallationToLicenseServer($request)
    {
        $license_server = 'https://springsoftit.com/api/install/addon';

        try {
            Http::timeout(60)->post($license_server, [
                'code' => $request->licence_key,
                'user' => $request->code_username,
                'domain' => request()->getHost(),
                'ip' => request()->ip(),
                'status' => 1,
                'type' => 2,
            ]);
        } catch (\Throwable $th) {
        }
    }
}
