<?php

namespace App\Http\Controllers\Backend\Blog;

use App\Filters\TitleFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Blog\BlogCreateRequest;
use App\Http\Requests\Backend\Blog\BlogUpdateRequest;
use App\Models\Blog;
use App\Models\BlogCategory;
use App\Services\Backend\Blog\BlogService;

class BlogController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:manage_blogs,admin');
    }

    function index()
    {

        $payloads['blogs'] = Blog::select('id','title','status','created_at','image')->latest()->filter([TitleFilter::class])->paginate(offsetPerPage());

        return view('backend.blogs.index')->with($payloads);
    }

    function create()
    {
        $payloads['categories'] = BlogCategory::select(['id', 'name'])
            ->latest()
            ->active()
            ->get();

        return view('backend.blogs.create')->with($payloads);
    }

    function store(BlogCreateRequest $blogCreateRequest, BlogService $blogService)
    {
        $payloads = $blogCreateRequest->validated();
        try {
            $response = $blogService->createBlog($payloads);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage())->withInput();
        }

        return redirectAfterOperation($response);
    }

    function edit($id)
    {
        $payloads['blog'] = Blog::findOrFail($id);
        $payloads['categories'] = BlogCategory::select(['id', 'name'])
        ->latest()
        ->active()
        ->get();

        return view('backend.blogs.edit')->with($payloads);
    }

    function update(BlogUpdateRequest $blogUpdateRequest, BlogService $blogService, $id)
    {
        $payloads = $blogUpdateRequest->validated();

        $purifier = new \HTMLPurifier();

        foreach ($payloads['description'] as $lang => $value) {
            foreach(languages() as $language){
                if($language->code == $lang){
                    $payloads['description'][$lang]['description'] =  htmlspecialchars_decode($purifier->purify($payloads['description'][$lang]['description']));
                }
            }
        }

        $blog = Blog::findOrFail($id);

        try {
            $response = $blogService->updateBlog($payloads, $blog);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage())->withInput();
        }

        return redirectAfterOperation($response);
    }

    function destroy(BlogService $blogService,$id)
    {
        $blog = Blog::findOrFail($id);

        try {
            $response = $blogService->delete($blog);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

       return redirectAfterOperation($response);
    }
}
