<?php

namespace App\Http\Controllers\Backend\Currency;

use App\Filters\NameFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Currency\StoreRequest;
use App\Http\Requests\Backend\Currency\UpdateRequest;
use App\Models\Currency;
use App\Models\Gateway;
use App\Services\Backend\Currency\CurrencyService;
use Illuminate\Support\Facades\DB;

class CurrencyController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:manage_currency,admin');
    }
    public function index()
    {
        $payloads['currencies'] = Currency::latest()->filter([NameFilter::class])->paginate(offsetPerPage());
        return view('backend.currency.index')->with($payloads);
    }

    public function create()
    {
        $payloads['gateways'] = Gateway::active()->get();
        return view('backend.currency.create')->with($payloads);
    }

    public function store(StoreRequest $request, CurrencyService $service)
    {
        $payloads = $request->validated();

        try {
            DB::beginTransaction();
            $response = $service->addCurrency($payloads);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }

    public function edit(Currency $currency)
    {
        $payloads['currency'] = $currency;
        $payloads['gateways'] = Gateway::active()->get();
        return view('backend.currency.edit')->with($payloads);
    }

    public function update(UpdateRequest $request, CurrencyService $service, Currency $currency)
    {
        $payloads = $request->validated();
        try {
            DB::beginTransaction();
            $response = $service->updateCurrency($payloads, $currency);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }

    public function destroy(CurrencyService $service, Currency $currency)
    {
        try {
            DB::beginTransaction();
            $response = $service->deleteCurrency($currency);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }
}
