<?php

namespace App\Http\Controllers\Backend\Language;

use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Language\StoreRequest;
use App\Http\Requests\Backend\Language\TranslationUpdateRequest;
use App\Http\Requests\Backend\Language\UpdateRequest;
use App\Models\Language;
use App\Services\Backend\Language\LanguageService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LanguageController extends Controller
{

    function __construct(){
        $this->middleware('permission:manage_language,admin');
    }

    function index()
    {
        $payloads['languages'] = Language::latest()->get();
        return view('backend.language.index')->with($payloads);
    }

    function store(StoreRequest $storeRequest, LanguageService $languageService)
    {
        $payloads = $storeRequest->validated();
        try {
            DB::beginTransaction();
            $response = $languageService->storeLanguage($payloads);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }
        return redirectAfterOperation($response);
    }

    function update(UpdateRequest $updateRequest, LanguageService $languageService, $id)
    {
        $payloads = $updateRequest->validated();
        $lang = Language::findOrFail($id);
        try {
            DB::beginTransaction();
            $response = $languageService->updateLanguage($payloads, $lang);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }
        return redirectAfterOperation($response);
    }

    function destroy($id, LanguageService $languageService)
    {
        $lang = Language::findOrFail($id);
        try {
            DB::beginTransaction();
            $response = $languageService->deleteLanguage($lang);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }
        return redirectAfterOperation($response);
    }

    function translate($code)
    {
        $payloads['lang'] = Language::where('code', $code)->firstOrFail();
        $payloads['backend'] = translations($payloads['lang'], 'backend.php','backend');
        $payloads['frontend'] = translations($payloads['lang'], 'frontend.php','frontend');
        $payloads['notification'] = translations($payloads['lang'], 'notification.php','notification');
        return view('backend.language.translate')->with($payloads);
    }


    function translateUpdate(TranslationUpdateRequest $translationUpdateRequest, LanguageService $languageService, $code){
        $payloads = $translationUpdateRequest->validated();
        $lang = Language::where('code', $code)->firstOrFail();
        try {
            $response = $languageService->updateTranslations($payloads, $lang);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }
        return redirectAfterOperation($response);
    }


    function syncTranslation(Request $request,LanguageService $languageService){
        $lang = Language::where('code', $request->code)->firstOrFail();
        $languageService->syncTranslation($lang);
    }


    function translationSearch(Request $request,LanguageService $languageService){
        $lang = Language::where('code', $request->code)->firstOrFail();
        $payloads['languages'] = $languageService->searchTranslation($lang, $request->type, $request->search);
        return view('backend.language.search-result')->with($payloads);
    }
}
