<?php

namespace App\Http\Controllers\Backend\Others;

use App\Constant\Status;
use App\Filters\EmailFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Editor\EditorImageRequest;
use App\Http\Requests\Backend\Subscribers\MailSendRequest;
use App\Models\Contact;
use App\Models\EmailSubscription;
use App\Models\Version;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class OthersController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:manage_subscribers,admin')->only('subscribers', 'subscriberBulkMail');
        $this->middleware('permission:manage_version,admin')->only('systemVersion');
    }

    function changeStatus(Request $request)
    {
        $class = Status::MODELS[$request->model];

        $column = $request->column;

        $model = new $class();

        $collection = $model->find($request->id);

        if (!$collection) {
            return ['status' => false, 'message' => tt_trans('Status not updated', cons()::LANGUAGE_NOTIFICATION)];
        }

        $collection->$column = !$collection->$column;

        $collection->save();

        clearCache();

        return ['status' => true, 'message' => tt_trans('Status updated Successfully', cons()::LANGUAGE_NOTIFICATION)];
    }

    function systemVersion()
    {
        $composerFile = base_path('composer.json');
        $composerData = json_decode(File::get($composerFile), true);
        $phpVersion = $composerData['require']['php'] ?? 'Not defined';
        
        
        clearCache();
        $payloads['php_version'] = $phpVersion;
        $payloads['laravel_version'] = app()->version();
        $payloads['timezone'] = config('app.timezone');
        $payloads['server_ip'] = $_SERVER['SERVER_ADDR'];
        $payloads['host'] = request()->getHttpHost();

        $payloads['updateVersions'] = Version::pluck('version')->toArray();

        return view('backend.others.system')->with($payloads);
    }

    function clearCache()
    {
        clearCache();

        return back()->with('success', tt_trans('Clear Cache Successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    function subscribers()
    {
        $payloads['subscribers'] = EmailSubscription::latest()
            ->filter([EmailFilter::class])
            ->paginate(offsetPerPage());

        return view('backend.subscribers.index')->with($payloads);
    }

    function subscriberBulkMail(MailSendRequest $mailSendRequest)
    {
        $payloads = $mailSendRequest->validated();

        try {
            $users = EmailSubscription::get();

            if ($users->isEmpty()) {
                return back()->with('error', tt_trans('No Subscribers Found', cons()::LANGUAGE_NOTIFICATION));
            }

            foreach ($users as $user) {
                sendMail($user->email, $payloads);
            }
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return back()->with('success', tt_trans('Bulk mail send successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    function subscriberMail(MailSendRequest $mailSendRequest, $id)
    {
        $payloads = $mailSendRequest->validated();

        try {
            $user = EmailSubscription::findOrFail($id);
            sendMail($user->email, $payloads);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return back()->with('success', tt_trans('Mail send successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    function editorImage(EditorImageRequest $editorImageRequest)
    {
        $payloads = $editorImageRequest->validated();

        if ($editorImageRequest->hasFile('file')) {
            $file = storeFile($payloads['file'], 'editor_image');

            return response()->json(['location' => getFile('editor_image', $file)]);
        }

        return response()->json(['error' => 'No image uploaded'], 400);
    }

    function markAsAllRead()
    {
        admin()->unreadNotifications()->update(['read_at' => now()]);

        return back()->with('success', tt_trans('All notification mark as read', cons()::LANGUAGE_NOTIFICATION));
    }

    function contact(){

        $payloads['contacts'] = Contact::latest()->paginate(offsetPerPage());
        
        return view('backend.contact.index')->with($payloads);
    }


    function fetchCryptoCurrencies()
    {
        $url = 'https://pro-api.coinmarketcap.com/v1/cryptocurrency/listings/latest';

        $parameters = [
            'start' => '1',
            'limit' => '3000',
            'convert' => 'USD',
        ];

        $headers = ['Accepts: application/json', 'X-CMC_PRO_API_KEY:'.tc()->get('crypto_api')];
        $qs = http_build_query($parameters); 
        $request = "{$url}?{$qs}";

        $curl = curl_init();
       
        curl_setopt_array($curl, [
            CURLOPT_URL => $request, 
            CURLOPT_HTTPHEADER => $headers, 
            CURLOPT_RETURNTRANSFER => 1,
        ]);

        $response = curl_exec($curl);


        return json_decode($response);
    }
}
