<?php

namespace App\Http\Controllers\Backend\Page;

use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Page\StoreRequest;
use App\Http\Requests\Backend\Page\UpdateRequest;
use App\Models\Page;
use App\Services\Backend\Page\PageService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class PagesController extends Controller
{

    function __construct(){
        $this->middleware('permission:manage_pages,admin');
    }


    function index()
    {
        $payloads['pages'] = Page::latest()->with('sections')->paginate(offsetPerPage());
        return view('backend.pages.index')->with($payloads);
    }

    function create()
    {
        $sections = sections()['sections'];

        $fixed_sections = sections()['fixed_position_sections'];

        $payloads['sections'] = array_diff_key($sections, array_flip($fixed_sections));
        
        return view('backend.pages.create')->with($payloads);
    }

    function store(StoreRequest $storeRequest, PageService $pageService)
    {
        $payloads = $storeRequest->validated();

        try {
            DB::beginTransaction();
            $response = $pageService->createPage($payloads);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        Cache::forget('pageContent');

        return redirectAfterOperation($response,'panel.page.index');
    }

    function edit(Page $page){
        $sections = sections()['sections'];

        $fixed_sections = sections()['fixed_position_sections'];

        $payloads['sections'] = array_diff_key($sections, array_flip($fixed_sections));
        $payloads['page'] = $page;
        return view('backend.pages.edit')->with($payloads);
    }

    function update(UpdateRequest $updateRequest, PageService $pageService, Page $page){
        $payloads = $updateRequest->validated();

        try {
            DB::beginTransaction();
            $response = $pageService->updatePage($payloads, $page);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        Cache::forget('pageContent');

        return redirectAfterOperation($response);
    }

    function destroy(PageService $pageService, Page $page){
        try {
            DB::beginTransaction();
            $response = $pageService->delete($page);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        Cache::forget('pageContent');

        return redirectAfterOperation($response);
    }
}
