<?php

namespace App\Http\Controllers\Backend\Payment;

use App\Filters\DateFilter;
use App\Filters\TrxFilter;
use App\Http\Controllers\Controller;
use App\Models\IcoSetting;
use App\Models\StagePurchase;
use App\Models\Transaction;
use App\Events\NotificationEvent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    public function pendingPurchases()
    {
        $payloads['payments'] = StagePurchase::pending()->filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        $payloads['title']  = "Pending Purchase";
        return view('backend.payments.purchase')->with($payloads);
    }
    public function rejectPurchase()
    {
        $payloads['payments'] = StagePurchase::rejected()->filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        $payloads['title'] = "Rejected Purchase";
        return view('backend.payments.purchase')->with($payloads);
    }

    public function completedPurchase()
    {
        $payloads['payments'] = StagePurchase::accepted()->filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        $payloads['title'] = "Completed Purchase";
        return view('backend.payments.purchase')->with($payloads);
    }

    public function offlinePaymentDetails($id)
    {
        $payloads['payment'] = StagePurchase::with(['user', 'method'])->findOrFail($id);
        $payloads['setting'] = IcoSetting::first();
        return view('backend.payments.details')->with($payloads);
    }

    public function offlinePaymentAccept($id)
    {
        $purchase = StagePurchase::pending()->whereId($id)->firstOrFail();

        DB::beginTransaction();

        $stage = $purchase->stage;
        $needToken = $purchase->token + $purchase->bonus_token;
        
        if ($stage->remainToken() < $needToken) {
            return back()->with('error', "Not enough token for this purchase");
        }

        $purchase->status = cons()::ACCEPTED;
        $purchase->save();




        $user = $purchase->user;
        $user->token += ($purchase->token + $purchase->bonus_token);
        $user->save();

        Transaction::create([
            'trx' => $purchase->trx,
            'user_id' => $user->id,
            'amount' => $purchase->total_price,
            'reason' => "Buy token",
            'type' => '-',
            'currency' => $purchase->currency->name,
            'formatter' => $purchase->currency->currency_type == cons()::CRYPTO ? 8 : 2
        ]);

        DB::commit();

        $message = "Purchase accepted successfully";
        event(new NotificationEvent($purchase, $user, $message, 'offline_purchase_accepted'));
        return back()->with('success', tt_trans('Payment accepted successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    public function offlinePaymentReject(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required'
        ]);

        $purchase = StagePurchase::pending()->whereId($id)->firstOrFail();

        DB::beginTransaction();

        $purchase->status = cons()::REJECTED;
        $purchase->note = $request->reason;
        $purchase->save();

        DB::commit();

        $user = $purchase->user;

        $message = "Purchase rejected";

        event(new NotificationEvent($purchase, $user, $message, 'offline_purchase_rejected'));
        return back()->with('success', tt_trans('Payment rejected successfully', cons()::LANGUAGE_NOTIFICATION));
    }
}
