<?php

namespace App\Http\Controllers\Backend\Stage;

use App\Filters\DateFilter;
use App\Filters\StageFilter;
use App\Filters\TrxFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\Stage\StoreRequest;
use App\Http\Requests\Backend\Stage\UpdateRequest;
use App\Models\IcoSetting;
use App\Models\Stage;
use App\Models\StagePurchase;
use App\Services\Backend\Stage\StageService;
use Illuminate\Support\Facades\DB;

class StageController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:manage_auction,admin');
    }
    public function index()
    {
        $payloads['stages'] = Stage::latest()->filter([StageFilter::class])->paginate(offsetPerPage());
        return view('backend.stage.index')->with($payloads);
    }

    public function create()
    {
        $payloads['setting'] = IcoSetting::first();

        if (!$payloads['setting']) {
            return redirect()->route('panel.ico.setting')->with('error', tt_trans('Please set ICO setting first', cons()::LANGUAGE_NOTIFICATION));
        }
        return view('backend.stage.create')->with($payloads);
    }

    public function store(StoreRequest $request, StageService $service)
    {
        $payloads = $request->validated();

        try {
            DB::beginTransaction();
            $response = $service->addStage($payloads);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }

    public function edit(Stage $stage)
    {
        $payloads['stage'] = $stage;
        $payloads['setting'] = IcoSetting::first();

        if (!$payloads['setting']) {
            return redirect()->route('panel.ico.setting')->with('error', tt_trans('Please set ICO setting first', cons()::LANGUAGE_NOTIFICATION));
        }
        return view('backend.stage.edit')->with($payloads);
    }

    public function update(UpdateRequest $request, StageService $service, Stage $stage)
    {
        $payloads = $request->validated();

        try {
            DB::beginTransaction();
            $response = $service->updateStage($payloads, $stage);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }

    public function destroy(StageService $service, Stage $stage)
    {
        try {
            DB::beginTransaction();
            $response = $service->deleteStage($stage);
            DB::commit();
        } catch (\Throwable $th) {
            return redirect()->back()->withErrors(['message' => $th->getMessage()]);
        }
        return redirectAfterOperation($response);
    }

    public function purchases()
    {
        $payloads['purchases'] = StagePurchase::filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        return view('backend.stage.purchases')->with($payloads);
    }
}
