<?php

namespace App\Http\Controllers\Backend\User;

use App\Filters\SortFilter;
use App\Filters\UserFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\User\AddBalanceRequest;
use App\Http\Requests\Backend\User\BulkMailRequest;
use App\Http\Requests\Backend\User\ProfileUpdateRequest;
use App\Http\Requests\Backend\User\SendEmailRequest;
use App\Http\Requests\Backend\User\SubBalanceRequest;
use App\Models\User;
use App\Services\Backend\User\UserService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;


class UserController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:manage_users,admin');
    }

    function users()
    {
        $payloads['users'] = User::latest()
            ->filter([UserFilter::class, SortFilter::class])
            ->paginate(offsetPerPage());

        $payloads['deactive'] = User::whereHas('ability', function ($item) {
            return $item->deactive();
        })->count();

        $payloads['email_unverified'] = User::whereHas('ability', function ($item) {
            $item->emailUnverified();
        })->count();

        $payloads['sms_unverified'] = User::whereHas('ability', function ($item) {
            $item->smsUnverified();
        })->count();

        $payloads['kyc_unverified'] = User::whereHas('ability', function ($item) {
            $item->kycUnverified();
        })->count();

        return view('backend.users.index')->with($payloads);
    }

    function userDetails($id)
    {
        $payloads['user'] = User::findOrFail($id);

        $payloads['transactions'] = $payloads['user']->transactions()->latest()->paginate(offsetPerPage(), ['*'], 'transactions');

        $payloads['supports'] = $payloads['user']->supports()->latest()->paginate(offsetPerPage(), ['*'], 'supports');

        $payloads['withdraws'] = $payloads['user']->withdraws()->latest()->paginate(offsetPerPage(), ['*'], 'withdraws');


        $payloads['dates'] = $payloads['user']->deposits()->select(DB::raw('DATE_FORMAT(created_at, "%d %b") as date'))
            ->union($payloads['user']->withdraws()->select(DB::raw('DATE_FORMAT(created_at, "%d %b") as date')))
            ->orderBy(DB::raw('STR_TO_DATE(date, "%d %b")'), 'ASC')
            ->pluck('date');

        $payloads['depositData'] = $payloads['user']->deposits()->accepted()->select(DB::raw('DATE_FORMAT(created_at, "%d %b") as date'), DB::raw('SUM(total_amount) as total_deposit'))->groupBy('date')->pluck('total_deposit', 'date');

        $payloads['withdrawData'] = $payloads['user']->withdraws()->accepted()->select(DB::raw('DATE_FORMAT(created_at, "%d %b") as date'), DB::raw('SUM(withdraw_amount) as total_withdraw'))->groupBy('date')->pluck('total_withdraw', 'date');

        $payloads['finalDepositData'] = [];
        $payloads['finalWithdrawData'] = [];

        foreach ($payloads['dates'] as $date) {
            $payloads['finalDepositData'][] = $payloads['depositData'][$date] ?? 0;
            $payloads['finalWithdrawData'][] = $payloads['withdrawData'][$date] ?? 0;
        }

        return view('backend.users.details')->with($payloads);
    }

    function bulkMail(BulkMailRequest $bulkMailRequest, UserService $userService)
    {
        $payloads = $bulkMailRequest->validated();

        try {
            $response = $userService->sendBulkMail($payloads);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response);
    }

    function deleteUser(User $user)
    {
        try {
            removeImage($user, 'users');
            $user->delete();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return back()->with('success', tt_trans('User deleted successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    function addBalance(AddBalanceRequest $addBalanceRequest, UserService $userService, User $user)
    {

        $payloads = $addBalanceRequest->validated();

        try {
            DB::beginTransaction();
            $response = $userService->addBalance($payloads, $user);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response);
    }


    function subBalance(SubBalanceRequest $subBalanceRequest, UserService $userService, User $user)
    {
        $payloads = $subBalanceRequest->validated();

        try {
            DB::beginTransaction();
            $response = $userService->subBalance($payloads, $user);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response);
    }

    function sendMail(SendEmailRequest $sendEmailRequest, UserService $userService, User $user)
    {
        $payloads = $sendEmailRequest->validated();

        try {
            $response = $userService->sendMail($payloads, $user);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response);
    }

    function updateUserProfile(ProfileUpdateRequest $profileUpdateRequest, UserService $userService, User $user)
    {
        $payloads = $profileUpdateRequest->validated();

        try {
            DB::beginTransaction();
            $response = $userService->updateUserProfile($payloads, $user);
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response);
    }

    function loginAsUser($id)
    {

        Auth::loginUsingId($id);

        return to_route('dashboard');
    }
}
