<?php

namespace App\Http\Controllers\Frontend\Auction;

use App\Filters\AuctionFilter;
use App\Filters\DateFilter;
use App\Filters\SellerFilter;
use App\Filters\TrxFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\Frontend\Auction\AuctionCreateRequest;
use App\Models\Auction;
use App\Events\NotificationEvent;
use App\Models\AuctionPurchase;
use App\Models\AuctionWithdraw;
use App\Models\IcoSetting;
use App\Models\Transaction;
use App\Services\Frontend\Auction\AuctionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AuctionController extends Controller
{
    public function create()
    {
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::auction.create')->with($payloads);
    }

    public function store(AuctionCreateRequest $request, AuctionService $service)
    {
        $payloads = $request->validated();

        try {
            DB::beginTransaction();
            $response = $service->store($payloads);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('auction.create')->with('error', tt_trans('An error occurred while creating the auction', cons()::LANGUAGE_NOTIFICATION));
        }

        return redirectAfterOperation($response, 'auctions');
    }

    public function auctions()
    {
        $payloads['auctions'] = Auction::whereUserId(user()->id)->filter([TrxFilter::class, AuctionFilter::class])->latest()->paginate(offsetPerPage());
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::auction.index')->with($payloads);
    }
    public function list()
    {
        $payloads['auctions'] = Auction::active()->where('user_id', '!=', user()->id)
            ->filter([SellerFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::auction.list')->with($payloads);
    }

    public function buy($trx)
    {
        $payloads['auction'] = Auction::active()->whereTrx($trx)->firstOrFail();

        if ($payloads['auction']->user_id == user()->id) {
            return redirect()->route('auction.list')->with('error', tt_trans('You cannot buy your own auction', cons()::LANGUAGE_NOTIFICATION));
        }
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::auction.buy')->with($payloads);
    }

    public function buyToken(Request $request, $trx)
    {
        $request->validate([
            'token' => 'required|gt:0|integer'
        ]);

        $auction = Auction::active()->whereTrx($trx)->firstOrFail();
        $setting = IcoSetting::first();

        if ($auction->user_id == user()->id) {
            return redirect()->route('auction.list')->with('error', tt_trans('You cannot buy your own auction', cons()::LANGUAGE_NOTIFICATION));
        }

        $token = $request->token;
        $price = $auction->token_base_price;
        $totalPrice = $token * $price;
        $adminGet = ($setting->auction_buy_percentage * $totalPrice) / 100;
        $sellerGet = $totalPrice - $adminGet;

        $user = user();
        $seller = $auction->user;

        if ($user->balance < $totalPrice) {
            return redirect()->back()->with('error', tt_trans('Insufficient balance', cons()::LANGUAGE_NOTIFICATION));
        }

        if ($auction->availableToken() < $token) {
            return redirect()->back()->with('error', tt_trans('Not enough token available', cons()::LANGUAGE_NOTIFICATION));
        }

        DB::beginTransaction();

        // seller part
        $seller->token -= $token;
        $seller->balance += $sellerGet;
        $seller->save();

        Transaction::create([
            'trx' => getTrx(),
            'user_id' => $seller->id,
            'amount' => $sellerGet,
            'reason' => "Sell token",
            'type' => '+',
            'currency' => tc()->get('currency'),
            'formatter' => 2
        ]);


        // buyer part

        $user->token += $token;
        $user->balance -= $totalPrice;
        $user->save();

        Transaction::create([
            'trx' => getTrx(),
            'user_id' => $user->id,
            'amount' => $totalPrice,
            'reason' => "Buy token",
            'type' => '-',
            'currency' => tc()->get('currency'),
            'formatter' => 2
        ]);

        $auction->sell_token += $token;
        $auction->save();

        // purchase log
        $purchase = new AuctionPurchase();
        $purchase->trx = getTrx();
        $purchase->auction_id = $auction->id;
        $purchase->seller_id = $seller->id;
        $purchase->buyer_id = $user->id;
        $purchase->tokens = $token;
        $purchase->base_price = $auction->token_base_price;
        $purchase->total_price = $totalPrice;
        $purchase->admin_profit = $adminGet;
        $purchase->save();

        DB::commit();

        // buyer notification

        $message = "Buy token successfully from " . $seller->username;
        event(new NotificationEvent($purchase, $user, $message, 'auction_buy_token'));

        // seller notification
        $message = $user->username . " purchase token ";
        event(new NotificationEvent($purchase, $seller, $message, 'auction_sell_token'));


        return redirect()->route('auction.list')->with('success', tt_trans('Auction successful', cons()::LANGUAGE_NOTIFICATION));
    }

    public function purchases()
    {
        $payloads['setting'] = IcoSetting::first();
        $payloads['purchases'] = AuctionPurchase::whereBuyerId(user()->id)->filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        return view('frontend::auction.purchases')->with($payloads);
    }

    public function sell()
    {
        $payloads['setting'] = IcoSetting::first();
        $payloads['sales'] = AuctionPurchase::whereSellerId(user()->id)->filter([TrxFilter::class, DateFilter::class])->latest()->paginate(offsetPerPage());
        return view('frontend::auction.sales')->with($payloads);
    }

    public function withdraw($trx)
    {
        $auction = Auction::whereUserId(user()->id)->active()->where('trx', $trx)->firstOrFail();
        $setting = IcoSetting::first();

        $auctionToken = $auction->tokens;
        $reduce = ceil(($setting->auction_refund_percentage * $auctionToken) / 100);

        if ($auction->availableToken() < $reduce) {
            return redirect()->back()->with('error', tt_trans('Not enough token available for withdraw', cons()::LANGUAGE_NOTIFICATION));
        }

        DB::beginTransaction();
        $auction->status = cons()::WITHDRAW_AUCTION;
        $auction->save();

        $refundable = $auctionToken - $reduce;
        $auction->user->token += ($refundable);
        $auction->save();

        $activeStage = getActiveStage()['stage'];
        if ($activeStage) {
            $activeStage->token_issues += $reduce;
            $activeStage->save();
        }

        $withdraw = new AuctionWithdraw();
        $withdraw->trx = getTrx();
        $withdraw->auction_id = $auction->id;
        $withdraw->refund_token = $refundable;
        $withdraw->withdraw_fine = $reduce;
        $withdraw->save();

        $user = $auction->user;

        $message = "Withdraw auction successfull";

        event(new NotificationEvent($withdraw, $user, $message, 'auction_withdraw'));


        DB::commit();

        return back()->with(['success', cons("Withdraw successfull", cons()::LANGUAGE_NOTIFICATION)]);
    }
}
