<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Http\Requests\Frontend\Contact\ContactRequestValidation;
use App\Http\Requests\Frontend\Subscribe\EmailSubscriptionRequest;
use App\Models\Blog;
use App\Models\BlogCategory;
use App\Models\Contact;
use App\Models\Currency;
use App\Models\EmailSubscription;
use App\Models\IcoSetting;
use App\Models\Page;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    function home()
    {
        $page = Page::where('name', 'home')->first();

        session()->forget('url.intended');

        $payloads['sections'] = $page->sections;

        return view('frontend::home')->with($payloads);
    }

    function page($slug)
    {
        $payloads['page'] = Page::active()->where('slug', $slug)->firstOrFail();

        return view('frontend::page.index')->with($payloads);
    }

    function subscribe(EmailSubscriptionRequest $emailSubscriptionRequest)
    {
        $payloads = $emailSubscriptionRequest->validated();

        try {
            EmailSubscription::create($payloads);
        } catch (\Throwable $th) {
            return ['status' => false, 'message' => tt_trans($th->getMessage(), cons()::LANGUAGE_NOTIFICATION)];
        }
        return ['status' => true, 'message' => tt_trans('Email subscription successfull', cons()::LANGUAGE_NOTIFICATION)];
    }

    function blog($tag = '')
    {
        $payloads['blogs'] = Blog::latest()
            ->active()
            ->when($tag, function ($query) use ($tag) {
                $query->whereJsonContains('tags', strtolower($tag))->orWhereJsonContains('tags', ucfirst($tag))->orWhereJsonContains('tags', strtoupper($tag));
            })
            ->paginate(offsetPerPage());
        $payloads['categories'] = BlogCategory::active()->whereHas('blogs', function ($query) {
            $query->active();
        })->get();
        $payloads['latestBlogs'] = Blog::active()->latest()->take(4)->get();

        $payloads['tags'] = Blog::active()->pluck('tags')->flatMap(function ($tags) {
            return $tags;
        })
            ->unique()
            ->values();

        return view('frontend::page.blog')->with($payloads);
    }
    function blogCategory($slug)
    {
        $category = BlogCategory::active()->where('slug', $slug)->firstOrFail();
        $payloads['blogs'] = $category->blogs()->latest()->active()->paginate(offsetPerPage());
        $payloads['latestBlogs'] = $category->blogs()->latest()->active()->take(4)->get();
        $payloads['categories'] = BlogCategory::active()->where('id', '!=', $category->id)
            ->whereHas('blogs', function ($query) {
                $query->active();
            })->get();

        $payloads['tags'] = Blog::pluck('tags')->flatten()->map(fn($tag) => strtolower($tag))->unique()->values();
        return view('frontend::page.blog')->with($payloads);
    }

    function blogDetails($slug)
    {
        $payloads['blog'] = Blog::where('slug', $slug)->firstOrFail();
        $payloads['latestBlogs'] = Blog::where('id', '!=', $payloads['blog']->id)
            ->latest()
            ->active()
            ->get();

        $payloads['categories'] = BlogCategory::whereHas('blogs', function ($query) {
            $query->active();
        })
            ->latest()
            ->take(10)
            ->get();

        $payloads['tags'] = Blog::pluck('tags')->flatten()->map(fn($tag) => strtolower($tag))->unique()->values();

        $payloads['shareComponent'] = \Share::page(route('blog.details', $slug), 'Share Link')->facebook()->twitter()->linkedin()->telegram()->whatsapp()->reddit();

        return view('frontend::page.blog_details')->with($payloads);
    }

    function contactView()
    {
        return view('frontend::page.contact');
    }

    function contact(ContactRequestValidation $contactRequestValidation)
    {
        try {
            Contact::create($contactRequestValidation->validated());
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return back()->with('success', tt_trans('Contact form submitted successfully', cons()::LANGUAGE_NOTIFICATION));
    }


    function stageCalculate(Request $request)
    {
        $stage = getActiveStage()['stage'];
        $data = $request->all();


        $currency = Currency::active()->whereId($data['convert_currency_id'])->first();
        $basePerTokenPrice = $stage->base_price;
        $baseTotalTokenPrice = $basePerTokenPrice * $data['get_token'];

        $convertRate = $baseTotalTokenPrice * (getActiveStage()['setting']->currency->exchange_rate / $currency->exchange_rate);

        $html = "
                <button type='button' class='currency-dropdown-btn'
                    data-converter-id='{$currency->id}'
                    data-bs-toggle='dropdown'>
                    <img src='" . getFile('currency', $currency->icon) . "' alt='image'>
                    <span>" . tt_front('Buy on ') . " {$currency->name} </span>
                    <i class='ri-arrow-down-s-line'></i>
                </button>";

        return [
            'status' => true,
            'selected_option' => $html,
            'base_price' => $currency->currency_type == cons()::CRYPTO ? formatter($convertRate / $data['get_token'], type: cons()::CRYPTO) : formatter($convertRate / $data['get_token']),
            'currency_symbol' => $currency->name,
            'pay_amount' => $currency->currency_type == cons()::CRYPTO ? formatter($convertRate, type: cons()::CRYPTO) : formatter($convertRate),
            'currency_id' => $currency->id
        ];
    }

    public function whitepaperDownload()
    {
        $setting = IcoSetting::first();


        if (!$setting || !$setting->white_paper) {
            return back()->with("error", tt_trans("No whitepaper found", cons()::LANGUAGE_NOTIFICATION));
        }

        $filePath = "files/image/white_paper/{$setting->white_paper}";
        if (!file_exists($filePath)) {
            return back()->with("error", tt_trans("No whitepaper found", cons()::LANGUAGE_NOTIFICATION));
        }

        return response()->download($filePath, 'Whitepaper.pdf', [
            'Content-Type' => 'application/pdf'
        ]);
    }
}
