<?php

namespace App\Http\Controllers\Frontend\Message;

use App\Events\MessageSent;
use App\Http\Controllers\Controller;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MessageController extends Controller
{
    public function messages()
    {
        $user = user();

        $payloads['senders'] =  Conversation::whereHas('messages')->where(function ($query) use ($user) {
            $query->where('user_one', $user->id)
                ->orWhere('user_two', $user->id);
        })->with(['messages' => function ($query) use ($user) {
            $query->where('sender_id', '!=', $user->id)
                ->latest();
        }, 'userOne', 'userTwo'])
            ->get()
            ->map(function ($conversation) use ($user) {

                return $conversation->user_one == $user->id
                    ? $conversation->userTwo
                    : $conversation->userOne;
            })
            ->unique('id');

        if(!$payloads['senders']->isNotEmpty())
        {
            return back()->with(['error'=>tt_trans("No message available",cons()::LANGUAGE_BACKEND)]);
        }

        return view('frontend::messages.index')->with($payloads);
    }

    public function send(Request $request)
    {
        try {
            // Validate the request
            $validatedData = $request->validate([
                'message' => 'required|string|max:1000',
                'receiver_id' => 'required|exists:users,id'
            ]);

            $userOne = user()->id;
            $userTwo = $validatedData['receiver_id'];

            $conversation = Conversation::findOrCreateConversation($userOne, $userTwo);
            $message = Message::create([
                'conversation_id' => $conversation->id,
                'sender_id' => $userOne,
                'message' => $validatedData['message']
            ]);


            broadcast(new MessageSent([
                'receiver_id' => $validatedData['receiver_id'],
                'data' => $message->toArray()
            ]))->toOthers();

            return response()->json([
                'status' => 'success',
                'message' => 'Message sent successfully!',
                'data' => $message
            ], 200);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {

            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong. Please try again later.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function oldMessages(Request $request)
    {
        $userOne = $request->user_one;
        $userTwo = $request->user_two;
        $isRead = $request->is_read ?? 0;

        if ($userOne == user()->id || $userTwo == user()->id) {
            $receiver = User::find($userTwo);

            $conversation = Conversation::findOrCreateConversation($userOne, $userTwo);

            if ($isRead == 1) {
                Message::where('conversation_id', $conversation->id)
                    ->where('sender_id', '!=', $userOne)
                    ->where('is_read', 0)
                    ->update([
                        'is_read' => 1
                    ]);
            }


            $messages = Message::where('conversation_id', $conversation->id)->orderBy('created_at', 'asc')->get();

            $user = user();

            $senders = Conversation::where(function ($query) use ($user) {
                $query->where('user_one', $user->id)
                    ->orWhere('user_two', $user->id);
            })
                ->with(['messages' => function ($query) use ($user) {
                    $query->where('sender_id', '!=', $user->id)
                        ->latest();
                }, 'userOne', 'userTwo'])
                ->whereHas('messages', function ($query) {
                    $query->latest('created_at'); // Order by latest message timestamp
                })
                ->get()
                ->sortByDesc(function ($conversation) {
                    return optional($conversation->messages->first())->created_at;
                })
                ->map(function ($conversation) use ($user) {
                    return $conversation->user_one == $user->id
                        ? $conversation->userTwo
                        : $conversation->userOne;
                })
                ->unique('id');

            $header = view("frontend::messages.header", compact('receiver'))->render();
            $body = view("frontend::messages.body", compact('messages', 'receiver'))->render();
            $sidebar = view("frontend::messages.sidebar", compact('senders', 'userTwo'))->render();

            return response()->json([
                'status' => 'success',
                'data' => [
                    "header" => $header,
                    "body" => $body,
                    'sidebar' => $sidebar
                ]
            ], 200);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized'
            ], 401);
        }
    }

    public function readMessages(Request $request)
    {
        $sender = $request->sender;
        $receiver = $request->receiver;
    }
}
