<?php

namespace App\Http\Controllers\Frontend\Payment;

use App\Factory\PaymentGatewayFactory;
use App\Http\Controllers\Controller;
use App\Http\Requests\Frontend\Deposit\BankRequest;
use App\Http\Requests\Frontend\Deposit\GatewayRequest;
use App\Models\Gateway;
use App\Models\StagePurchase;
use App\Services\Frontend\Payment\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{


    function paymentInitiate(GatewayRequest $paymentRequest, PaymentService $paymentService)
    {
        $payloads = $paymentRequest->validated();

        session()->forget('view_returned');

        try {
            $response = $paymentService->payment($payloads);
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response, 'payment-overview');
    }

    function paymentInstruction(Request $request)
    {
        $payloads['gateway'] = Gateway::findOrFail($request->method);
        return view('user::payment.fetch-instruction')->with($payloads);
    }

    function paymentOverview()
    {
        $transaction = session('transaction');

        if (!$transaction) {
            return redirect()->route('dashboard')->with('error', 'Transaction not started.');
        }

        $transaction['gateway'] = Gateway::find(session('transaction')['method_id']);

        session()->forget('view_returned');

        return view('user::payment.payment-overview')->with($transaction);
    }

    function paymentrequirements()
    {
        $transaction = session('transaction');

        if (!$transaction) {
            return redirect()->route('deposit')->with('error', 'Transaction not started.');
        }

        $transaction['gateway'] = Gateway::findOrFail($transaction['method_id']);

        return view('user::payment.requirements')->with($transaction);
    }

    function paymentConfirmation(BankRequest $bankRequest, PaymentService $paymentService)
    {
        $transaction = session('transaction');

        $method = Gateway::findOrFail($transaction['method_id']);

        try {
            if ($method->gateway_type == cons()::ONLINE_GATEWAY) {
                $gateway = PaymentGatewayFactory::create(strtolower($method->name));
                $response = $gateway->initPayment($method, $transaction);

                if ($response['type'] == 'response') {
                    return $response['redirect'];
                }
                return redirect()->to($response['redirect']);
            }

            DB::beginTransaction();
            $response = $paymentService->offlineConfirmation($bankRequest->validated());
            DB::commit();
        } catch (\Throwable $th) {
            return back()->with('error', $th->getMessage());
        }

        return redirectAfterOperation($response, 'payment-final');
    }

    function paymentFinal()
    {
        $payloads = session('transaction') ?? [];

        if (count($payloads) <= 0) {
            return to_route('dashboard')->with('error', tt_trans('Invalid Payment Request', cons()::LANGUAGE_NOTIFICATION));
        }

        $gateway = Gateway::findOrFail($payloads['method_id']);

        $payment = StagePurchase::where('trx', $payloads['trx'])->firstOrFail();

        return view('user::payment.payment-final', compact('gateway', 'payment'));
    }
}
