<?php

namespace App\Http\Controllers\Frontend\Stage;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use App\Models\Admin;
use App\Models\IcoSetting;
use App\Models\Stage;
use App\Models\StagePurchase;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Events\NotificationEvent;
use App\Filters\PaymentFilter;
use App\Filters\TrxFilter;
use Illuminate\Support\Facades\DB;

class StageController extends Controller
{
    public function index()
    {
        $payloads['stages'] = Stage::isNotInActive()->get();
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::stage.index')->with($payloads);
    }

    public function purchaseStage()
    {
        $payloads['purchases'] = StagePurchase::latest()->where('user_id', user()->id)->filter([TrxFilter::class, PaymentFilter::class])->paginate(offsetPerPage());
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::stage.purchase')->with($payloads);
    }

    public function buyStage($id)
    {
        $payloads['stage'] = Stage::isNotInActive()->whereId($id)->firstOrFail();
        if (!stageStatus(($payloads['stage']->id))['is_buy']) {
            return redirect()->route('stage.index')->with(['error', tt_trans('Stage not buyable right now', cons()::LANGUAGE_NOTIFICATION)]);
        }
        $payloads['crypto'] = Currency::crypto()->active()->get();
        $payloads['fiat'] = Currency::fiat()->active()->get();
        $payloads['setting'] = IcoSetting::first();
        return view('frontend::stage.buy')->with($payloads);
    }

    public function calculation(Request $request)
    {
        $data = $request->all();

        $stage = Stage::active()->whereId($data['stage'])->first();
        $currency = Currency::active()->whereId($data['currency'])->first();

        $token = $data['get_token'];

        $basePerTokenPrice = $stage->base_price;
        $baseTotalTokenPrice = $basePerTokenPrice * $token;

        if ($currency) {
            $convertRate = $baseTotalTokenPrice * (getActiveStage()['setting']->currency->exchange_rate / $currency->exchange_rate);
        } else {
            $convertRate = $baseTotalTokenPrice;
        }



        if ($currency) {

            $html = "<img src=" . getFile('currency', $currency->icon ?? tc()->get('currency')) . "
                                                    alt='image' class='rounded-circle'>" . $currency->name ?? tc()->get('currency');

            return response()->json([
                'payable_amount' => $currency->currency_type == cons()::CRYPTO ? formatter($convertRate, type: cons()::CRYPTO) : formatter($convertRate),
                'html' => $html,
                'payment_route' => route('stage.buy.payment', ['stage' => $stage, 'currency' => $currency->id])
            ]);
        } else {
            return response()->json([
                'payable_amount' => $convertRate,
                'html' => "",
                'payment_route' => ""
            ]);
        }
    }

    public function withBalance(Request $request, $id)
    {

        $request->validate([
            'hidden_currency_id' => 'required',
            'hidden_quantity' => 'required|integer|gt:0'
        ]);

        $data = $request->all();

        $stage = Stage::active()->whereId($id)->firstOrFail();
        $token = $data['hidden_quantity'];


        if ($stage->soft_cap > $token || $stage->hard_cap < $token) {
            return back()->with('error', 'Invalid token quantity');
        }

        if ($stage->remainToken() < $data['hidden_quantity']) {
            return back()->with('error', 'Not enough tokens');
        }

        $price = $stage->base_price * $token;

        $user = user();

        if ($price > $user->balance) {
            return back()->with('error', 'Insufficient balance');
        }

        DB::beginTransaction();
        $purchase = new StagePurchase();
        $purchase->trx = getTrx();
        $purchase->stage_id = $id;
        $purchase->user_id = $user->id;
        $purchase->token = $token;
        $purchase->bonus_token = $stage->bonus_token;
        $purchase->base_price = $stage->base_price;
        $purchase->total_price = $price;
        $purchase->status = cons()::ACCEPTED;
        $purchase->payment_type = cons()::WITH_BALANCE;
        $purchase->save();

        $user->balance -= $price;
        $user->token += ($token + $stage->bonus_token);
        $user->save();

        Transaction::create([
            'trx' => $purchase->trx,
            'user_id' => $user->id,
            'amount' => $price,
            'reason' => "Buy token",
            'type' => '-',
            'currency' => tc()->get('currency'),
            'formatter' => 2
        ]);

        $message = "Successful purchase token";

        event(new NotificationEvent($purchase, $user, $message, 'purchase_token'));

        $message = $user->username . " Token purchase successfully";
        $admin = Admin::first();
        event(new NotificationEvent($purchase, $admin, $message, 'purchase_token_create'));

        DB::commit();

        return redirect()->route('purchase.stage')->with('success', tt_trans('Token purchased successfully', cons()::LANGUAGE_NOTIFICATION));
    }

    public function withPayment(Request $request, $stage, $currency)
    {

        $data = $request->validate([
            'hidden_quantity' => 'required|integer|gt:0'
        ]);

        $stage = Stage::active()->whereId($stage)->firstOrFail();
        $currency = Currency::active()->whereId($currency)->firstOrFail();

        if ($stage->soft_cap > $data['hidden_quantity'] || $stage->hard_cap < $data['hidden_quantity']) {
            return back()->with('error', 'Invalid token quantity');
        }

        if ($stage->remainToken() < $data['hidden_quantity']) {
            return back()->with('error', 'Not enough tokens');
        }


        $payloads['methods'] = $currency->gateways;
        $price = $data['hidden_quantity'] * $stage->base_price;
        $convertRate = $price * (getActiveStage()['setting']->currency->exchange_rate / $currency->exchange_rate);

        session()->put('stage_buy_informations', [
            'stage_id' => $stage->id,
            'price' => $currency->currency_type == cons()::CRYPTO ? formatter($convertRate, cons()::CRYPTO) : formatter($convertRate),
            'token' => $data['hidden_quantity'],
            'currency_id' => $currency->id
        ]);
        $payloads['data'] = session('stage_buy_informations');
        $payloads['currency'] = $currency;

        return view('user.payment.index')->with($payloads);
    }
}
