<?php
namespace App\Http\Controllers\Installer;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class InstallerController extends Controller
{
    private $permissions = ['manage_role', 'manage_staffs', 'manage_settings', 'manage_email', 'manage_referral', 'manage_language', 'manage_withdraw', 'manage_pages', 'manage_menus', 'manage_gateways', 'manage_users', 'manage_blogs', 'manage_kyc_request', 'manage_support', 'manage_deposits', 'manage_notifications', 'manage_subscribers', 'manage_frontend', 'manage_theme', 'manage_reports', 'manage_version', 'manage_currency', 'manage_ico_setting', 'manage_stage', 'manage_auction'];


    public function __construct()
    {
        $this->middleware('web');
    }

    function installer()
    {
        Artisan::call('optimize:clear');

        return view('installer.index');
    }

    function extensions()
    {
        $config = ['BCMath', 'Ctype', 'Fileinfo', 'JSON', 'Mbstring', 'OpenSSL', 'PDO', 'pdo_mysql', 'Tokenizer', 'XML', 'cURL', 'GD'];

        $procceedtoNextPage = true;

        return view('installer.extensions', compact('config', 'procceedtoNextPage'));
    }

    function permissions()
    {
        $config = ['BCMath', 'Ctype', 'Fileinfo', 'JSON', 'Mbstring', 'OpenSSL', 'PDO', 'pdo_mysql', 'Tokenizer', 'XML', 'cURL', 'GD'];

        if (!version_compare(PHP_VERSION, '8.1', '>=')) {
            return to_route('extensions')->with('error', 'Please check php version');
        }

        foreach ($config as $conf) {
            if (!checkExtenstion($conf)) {
                return to_route('extensions')->with('error', 'Please fill up all requirements');
            }
        }

        $permissions = ['project/bootstrap/cache/', 'project/storage/', 'project/storage/app/', 'project/storage/framework/', 'project/storage/logs/'];

        $procceedtoNextPage = true;

        return view('installer.permissions', compact('permissions', 'procceedtoNextPage'));
    }

    function database()
    {

        $permissions = ['project/bootstrap/cache/', 'project/storage/', 'project/storage/app/', 'project/storage/framework/', 'project/storage/logs/'];

        foreach ($permissions as $permission) {
            if (!folderPermissionCheck($permission)) {
                return to_route('extensions')->with('error', 'Please fill up all requirements');
            }
        }

        return view('installer.database');
    }

    function install(Request $request)
    {
        $payloads = $request->validate([
            'licence_key' => 'required',
            'code_username' => 'required',
            'app_name' => 'required',
            'site_url' => 'required|url',
            'database_name' => 'required',
            'database_username' => 'required',
            'database_host' => 'required',
            'database_password' => 'nullable',
            'database_port' => 'required|integer|between:1,65535',
        ]);

        if (!$this->checkDatabaseConnection($payloads)) {
            return back()->withInput()->with('error', 'Could not connect to database');
        }

        $this->updateEnvironmentAfterInstalltion($payloads);

        return to_route('credentials')->with('success', 'Successfully Installed project');
    }


    function verifyPurchase($purchase_code)
    {
        $personalToken = 'PbldGXTOooIBQmOfAOn0VuvfASGe6C2n';

        $code = trim($purchase_code);
        
        $expectedProductId = 48918061;

        if (!preg_match("/^([a-f0-9]{8})-(([a-f0-9]{4})-){3}([a-f0-9]{12})$/i", $code)) {
            return ['status' => false, 'message' => 'Invalid purchase code'];
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => "https://api.envato.com/v3/market/author/sale?code={$code}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_HTTPHEADER => ["Authorization: Bearer {$personalToken}", 'User-Agent: Purchase code verification script'],
        ]);

        $response = @curl_exec($ch);
        $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch) > 0) {
            return ['status' => false, 'message' => 'Failed to connect: ' . curl_error($ch)];
        }

        switch ($responseCode) {
            case 404:
                return ['status' => false, 'message' => 'Invalid purchase code'];
            case 403:
                return ['status' => false, 'message' => 'The personal token is missing the required permission for this script'];
            case 401:
                return ['status' => false, 'message' => 'The personal token is invalid or has been deleted'];
        }

        if ($responseCode !== 200) {
            return ['status' => false, 'message' => "Got status {$responseCode}, try again shortly"];
        }

        $body = @json_decode($response);

        if ($body === false && json_last_error() !== JSON_ERROR_NONE) {
            return ['status' => false, 'message' => 'Error parsing response, try again'];
        }

        if (!isset($body->item->id) || $body->item->id != $expectedProductId) {
            return ['status' => false, 'message' => 'This purchase code is not valid for your product'];
        }

        return ['status' => true, 'message' => $body];
    }

    function checkLicense($purchase_code, $user, $current_domain)
    {
        
        $license_server = 'https://springsoftit.com/api/installations';

        try {
            $response = Http::post($license_server, [
                'code' => $purchase_code,
                'user' => $user,
                'domain' => $current_domain,
                'ip' => request()->ip(),
            ]);
    
            return $response->json();
           
        } catch (\Throwable $th) {
            //throw $th;
        }


        return ['status' => false,'message' => 'Could not connect to server'];

    }

    private function migrate()
    {
        try {
            Schema::disableForeignKeyConstraints();
            $tables = DB::select('SHOW TABLES');
            $database = env('DB_DATABASE');
            $keyName = "Tables_in_{$database}";

            foreach ($tables as $table) {
                $tableName = $table->$keyName;
                DB::statement("DROP TABLE IF EXISTS `$tableName`");
            }
            Schema::enableForeignKeyConstraints();
            $sqlFilePath = base_path('database/sql/database.sql');
            $sql = file_get_contents($sqlFilePath);

            if ($sql === false) {
                throw new Exception('Unable to read the SQL file.');
            }
            DB::unprepared($sql);
            return $this->response('Database imported successfully.', 'success');
        } catch (Exception $e) {
            return $this->response($e->getMessage(), 'error');
        }
    }

    private function response($message, $status)
    {
        return [
            'status' => $status,
            'message' => $message,
        ];
    }

    function credentials()
    {
        if (env('DB_USERNAME') == '' || env('DB_HOST') == '' || env('DB_PORT') == '') {
            return to_route('database')->with('error', 'Database credentials are required')->withInput();
        }

        return view('installer.credentials');
    }

    function credentialStore(Request $request)
    {
        $request->validate([
            'admin_username' => 'required',
            'admin_email' => 'required|email',
            'admin_password' => 'required',
        ]);

        $this->migrate();


        $admin = Admin::create([
            'username' => $request->admin_username,
            'email' => $request->admin_email,
            'password' => bcrypt($request->admin_password),
            'status' => cons()::ACTIVE,
            'is_default' => true,
        ]);

        $role = Role::create(['name' => 'Admin', 'guard_name' => 'admin']);

        $admin->assignRole([$role->name]);

        foreach ($this->permissions as $permission) {
            Permission::create(['name' => $permission, 'guard_name' => 'admin']);
        }

        $permissions = Permission::pluck('id');

        $role->syncPermissions($permissions);

        $config = require base_path('licence.php');

        $config['is_installed'] = 'installed';
        $config['installation_time'] = now()->toDateString();
        $config['installation_server'] = env('APP_URL');
        $config['installation_ip'] = $request->ip();

        $update = var_export($config, true);

        $newContent = "<?php\n\nreturn " . $update . ";\n";

        file_put_contents(base_path('licence.php'), $newContent);

        $expireLink = URL::temporarySignedRoute('install-success', now()->addMinutes(1));

        return redirect()->to($expireLink);
    }

    protected function copyAndReplaceFile($sourceFilePath, $destinationFilePath)
    {
        if (file_exists($sourceFilePath)) {
            $fileContents = file_get_contents($sourceFilePath);

            file_put_contents($destinationFilePath, $fileContents);
        }
    }

    function updateEnvironmentAfterInstalltion($payloads)
    {
        $envFile = 'project/.env';

        if (file_exists('project/.env')) {
            $envFile = 'project/.env';
        } else {
            $newFile = fopen('project/.env', 'w');
            $envFile = 'project/.env';
        }

        if (file_exists($envFile)) {
            $envContent = '';
            $envContent .= 'APP_NAME=' . $payloads['app_name'] . "\n";
            $envContent .= 'APP_ENV=local' . "\n";
            $envContent .= 'APP_KEY=' . 'base64:Tqay8bZS9ZDrrmpRf92UT2s3H8XGq2OdUvcmgacoeWI=' . "\n";
            $envContent .= 'APP_DEBUG=false' . "\n";
            $envContent .= 'APP_URL=' . $payloads['site_url'] . "\n";
            $envContent .= 'DB_DATABASE= ' . '"' . $payloads['database_name'] . '"' . "\n";
            $envContent .= 'DB_USERNAME = ' . '"' . $payloads['database_username'] . '"' . "\n";
            $envContent .= 'DB_PASSWORD = ' . '"' . $payloads['database_password'] . '"' . "\n";
            $envContent .= 'DB_HOST = ' . '"' . $payloads['database_host'] . '"' . "\n";
            $envContent .= 'DB_PORT = ' . '"' . $payloads['database_port'] . '"' . "\n";
            $envContent .= 'QUEUE_CONNECTION = ' . 'database' . "\n";
            $envContent .= 'TIMEZONE = ' . 'Asia/Dhaka' . "\n";
            $envContent .= 'LICENSE_KEY = ' . '"' . $payloads['licence_key'] . '"' . "\n";
        }

        file_put_contents($envFile, $envContent);
    }

    function installSuccess()
    {
        return view('installer.thank-you');
    }

    private function checkDatabaseConnection($payloads)
    {
        $settings = config('database.connections.mysql');

        config([
            'database' => [
                'default' => 'mysql',
                'connections' => [
                    'mysql' => array_merge($settings, [
                        'driver' => 'mysql',
                        'host' => $payloads['database_host'],
                        'port' => $payloads['database_port'],
                        'database' => $payloads['database_name'],
                        'username' => $payloads['database_username'],
                        'password' => $payloads['database_password'],
                    ]),
                ],
            ],
        ]);

        $settings = config('database.connections');

        DB::purge();

        try {
            DB::connection()->getPdo();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}
