<?php

namespace App\Models;

use App\Traits\Filterable;
use Illuminate\Database\Eloquent\Model;

class Stage extends Model
{
    use Filterable;
    protected $guarded = [];

    function scopeActive($query)
    {
        return $query->where('status', cons()::ACTIVE);
    }

    function scopeIsNotInActive($query)
    {
        return $query->where('status', '!=', cons()::INACTIVE);
    }

    function scopeRunningStage($query)
    {
        return $query->isNotInActive()
            ->where(function ($q) {
                $q->where('start_date', '<=', now())
                    ->where('end_date', '>=', now());
            });
    }

    function exchangeRate($currency)
    {
        $stage = $this;
        $value = $stage->base_price / $currency->exchange_rate;

        return $currency->currency_type == cons()::CRYPTO ? formatter($value, type: cons()::CRYPTO) : formatter($value);
    }

    function purchases()
    {
        return $this->hasMany(StagePurchase::class);
    }

    function purchaseToken()
    {
        return $this->purchases()
            ->where('status', cons()::ACCEPTED)
            ->selectRaw('SUM(token + bonus_token) as total_token')
            ->value('total_token') ?? 0;
    }

    function remainToken()
    {
        return $this->token_issues - $this->purchaseToken();
    }

    function tokenSellPercentage()
    {
        $totalToken = $this->token_issues;
        $sellToken = (float) $this->purchaseToken();
        $percentage = ($sellToken / $totalToken) * 100;
        return number_format(($sellToken / $totalToken) * 100, 2);
    }
}
