<?php

namespace App\Models;

use App\Traits\Filterable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, Filterable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $guarded = [];

    protected $casts = [
        'password' => 'hashed',
        'send_at' => 'datetime',
    ];





    function referrals()
    {
        return $this->hasMany(UserReferral::class);
    }

    function myCommissions()
    {
        return $this->hasMany(ReferralCommission::class, 'who_received');
    }

    function initiateReferral()
    {
        return $this->hasMany(ReferralCommission::class, 'who_initiate');
    }

    function scopeGiveAway($query, $user)
    {
        return $this->initiateReferral()->where('who_received', $user)->sum('amount');
    }

    function referredUser()
    {
        return $this->hasOne(UserReferral::class, 'referred_user_id');
    }

    function unreadMessage()
    {
        return $this->hasMany(Message::class, 'sender_id')
            ->whereHas('conversation', function ($query) {
                $query->where('user_one', user()->id)
                    ->orWhere('user_two', user()->id);
            })
            ->where('is_read', 0)
            ->count();
    }

    function ability()
    {
        return $this->hasOne(UserAbility::class)->withDefault();
    }

    function address()
    {
        return $this->hasOne(UserAddress::class)->withDefault();
    }

    function kycs()
    {
        return $this->hasMany(KycRequest::class);
    }

    function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    function supports()
    {
        return $this->hasMany(Support::class);
    }

    function deposits()
    {
        return $this->hasMany(Deposit::class);
    }

    function withdraws()
    {
        return $this->hasMany(Withdraw::class);
    }

    function scopeTodayWithdraws($query)
    {
        return $query->whereHas('withdraws', function ($item) {
            return $item->whereBetween('created_at', [now()->startOfDay(), now()->endOfDay()]);
        });
    }

    function earnings()
    {
        return $this->hasMany(UserEarnings::class);
    }

    function scopeActive($query)
    {
        return $query->whereHas('ability', function ($item) {
            $item->active();
        });
    }

    function scopeInactive($query)
    {
        return $query->whereHas('ability', function ($item) {
            $item->deactive();
        });
    }

    function scopeEmailUnverified($query)
    {
        return $query->whereHas('ability', function ($item) {
            $item->emailUnverified();
        });
    }

    function getFullAddress()
    {
        return $this->address->city . ',' . $this->address->state . ',' . $this->address->zip . ',' . $this->address->country;
    }

    public function getFullNameAttribute()
    {
        return "{$this->f_name} {$this->l_name}";
    }

    public function sales()
    {
        return $this->hasMany(AuctionPurchase::class, 'seller_id');
    }

    public function purchases()
    {
        return $this->hasMany(AuctionPurchase::class, 'buyer_id');
    }

    public function getEmailAttribute()
    {
        $email = $this->attributes['email'];

        if (config('app.restriction_for_action')) {
            list($username, $domain) = explode('@', $email);

            if (strlen($username) > 2) {
                $maskedUsername = substr($username, 0, 1) . str_repeat('*', strlen($username) - 2) . substr($username, -1);
            } else {
                $maskedUsername = $username;
            }

            return $maskedUsername . '@' . $domain;
        }
        return $email;
    }
}
