<?php

namespace App\Services\Backend\Language;

use App\Models\Language;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

class LanguageService
{
    function storeLanguage(array $payloads)
    {
        if (isset($payloads['image'])) {
            $payloads['image'] = storeFile($payloads['image'], 'language');
        }

        $payloads['code'] = strtolower($payloads['code']);

        $lang = Language::create($payloads);

        $this->createLanguageFile($lang->code, 'backend.php');
        $this->createLanguageFile($lang->code, 'frontend.php');
        $this->createLanguageFile($lang->code, 'notification.php');


        Cache::forget('language');

        return ['status' => true, 'message' => tt_trans('Language created successfully', cons()::LANGUAGE_NOTIFICATION)];
    }

    function updateLanguage(array $payloads, $lang)
    {
        if (isset($payloads['image'])) {
            $payloads['image'] = storeFile($payloads['image'], 'language', $lang->image);
        }

        $lang->update($payloads);

        Cache::forget('language');

        return ['status' => true, 'message' => tt_trans('Language updated successfully', cons()::LANGUAGE_NOTIFICATION)];
    }

    function syncTranslation($lang)
    {
        $backendTranslations = require resource_path('lang/en/backend.php');
        $frontendTranslations = require resource_path('lang/en/frontend.php');
        $notificationTranslations = require resource_path('lang/en/notification.php');

        $currentBackendTranslations = require resource_path("lang/{$lang->code}/backend.php");
        $currentFrontendTranslations = require resource_path("lang/{$lang->code}/frontend.php");
        $currentNotificationTranslations = require resource_path("lang/{$lang->code}/notification.php");

        $backend = $currentBackendTranslations + $backendTranslations;
        $frontend = $currentFrontendTranslations + $frontendTranslations;
        $notification = $currentNotificationTranslations + $notificationTranslations;

        $backendFilePath = resource_path("lang/{$lang->code}/backend.php");
        $frontendFilePath = resource_path("lang/{$lang->code}/frontend.php");
        $notificationFilePath = resource_path("lang/{$lang->code}/notification.php");

        $backendContent = '<?php' . PHP_EOL . PHP_EOL . 'return ' . var_export($backend, true) . ';' . PHP_EOL;

        file_put_contents($backendFilePath, $backendContent);

        $frontendContent = '<?php' . PHP_EOL . PHP_EOL . 'return ' . var_export($frontend, true) . ';' . PHP_EOL;

        file_put_contents($frontendFilePath, $frontendContent);

        $notificationContent = '<?php' . PHP_EOL . PHP_EOL . 'return ' . var_export($notification, true) . ';' . PHP_EOL;

        file_put_contents($notificationFilePath, $notificationContent);
    }

    function searchTranslation($lang, $type, $search)
    {
        $language = $lang->code;
        $filePath = resource_path("lang/{$language}/{$type}.php");

        $messagesArray = require $filePath;
        $messagesCollection = collect($messagesArray);

        $query = $search;
        $results = $messagesCollection->filter(function ($value, $key) use ($query) {
            return stripos($key, $query) !== false;
        });

        return $results;
    }

    function deleteLanguage($lang)
    {
        $directoryPath = resource_path("lang/{$lang->code}");

        if (File::isDirectory($directoryPath)) {
            File::deleteDirectory($directoryPath);
        }

        removeImage($lang, 'language');

        if (session('language') == $lang->code) {
            session()->forget('language');
        }

        $lang->delete();

        Cache::forget('language');

        return ['status' => true, 'message' => tt_trans('Language deleted successfully', cons()::LANGUAGE_NOTIFICATION)];
    }

    function updateTranslations(array $payloads, $lang)
    {
        $language = $lang->code;

        $filePath = resource_path("lang/{$language}/{$payloads['name']}.php");

        $oldTranslations = require $filePath;

        $newTranslations = $payloads['translations'] + $oldTranslations;

        $fileContent = '<?php' . PHP_EOL . PHP_EOL . 'return ' . var_export($newTranslations, true) . ';' . PHP_EOL;

        file_put_contents($filePath, $fileContent);

        return ['status' => true, 'message' => tt_trans('Language updated successfully', cons()::LANGUAGE_NOTIFICATION)];
    }

    function createLanguageFile($lang, $filename)
    {
        $sourceFilePath = resource_path("lang/en/{$filename}");

        $destinationDirectoryPath = resource_path("lang/{$lang}");

        $destinationFilePath = "{$destinationDirectoryPath}/{$filename}";

        if (File::exists($sourceFilePath)) {
            $content = require $sourceFilePath;

            if (!File::isDirectory($destinationDirectoryPath)) {
                File::makeDirectory($destinationDirectoryPath, 0755, true);
            }

            $fileContent = '<?php' . PHP_EOL . PHP_EOL . 'return ' . var_export($content, true) . ';' . PHP_EOL;

            File::put($destinationFilePath, $fileContent);
        }
    }
}
