<?php

namespace App\Utils;

use Illuminate\Support\Facades\File;
use ReflectionClass;

class CronManager
{
    private $moduleStatus;

    public function __construct()
    {
        $this->moduleStatus = $this->loadModuleStatus();
    }

    private function loadModuleStatus()
    {
        $filePath = base_path('modules_statuses.json');

        if (File::exists($filePath)) {
            return json_decode(File::get($filePath), true);
        }

        return [];
    }

    public function getCrons()
    {
        $cronClasses = [];
        $cronFolders = $this->findCronFolders(base_path());

        foreach ($cronFolders as $folder) {
           $namespace = $this->getNamespaceFromPath($folder);
            if ($namespace) {
                $cronClasses = array_merge($cronClasses, $this->loadCronClassesFromNamespace($namespace, $folder));
            }
            
        }
        
        return $cronClasses;
    }

    private function findCronFolders($path)
    {
        $cronFolders = [];
        foreach (File::directories($path) as $directory) {
            if (strpos($directory, base_path('vendor')) === 0) {
                continue; // Skip vendor directories
            }

            if (basename($directory) === 'Modules') {
                // Iterate through each module in the Modules directory
                foreach (File::directories($directory) as $moduleDirectory) {
                    $moduleName = basename($moduleDirectory);
                    if (isset($this->moduleStatus[$moduleName]) && !$this->moduleStatus[$moduleName]) {
                        continue; // Skip inactive modules
                    }

                    // Check for Cron folder inside the module
                    foreach (File::directories($moduleDirectory) as $subDirectory) {
                        if (basename($subDirectory) === 'Cron') {
                            $cronFolders[] = $subDirectory;
                        }
                    }
                }
                continue; // Skip further processing of the Modules directory
            }

            if (basename($directory) === 'Cron') {
                $cronFolders[] = $directory;
            }

            // Recursively scan subdirectories
            $cronFolders = array_merge($cronFolders, $this->findCronFolders($directory));
        }

        return $cronFolders;
    }

    private function getNamespaceFromPath($folderPath)
    {
        $composerJson = json_decode(File::get(base_path('composer.json')), true);

        if (!isset($composerJson['autoload']['psr-4'])) {
            return null;
        }

        foreach ($composerJson['autoload']['psr-4'] as $namespace => $path) {
            $absolutePath = realpath(base_path($path));

            if ($absolutePath && str_starts_with(realpath($folderPath), $absolutePath)) {
                $relativePath = str_replace($absolutePath, '', realpath($folderPath));
                $relativeNamespace = str_replace(DIRECTORY_SEPARATOR, '\\', trim($relativePath, DIRECTORY_SEPARATOR));
                return $namespace . $relativeNamespace;
            }
        }

        return null;
    }

    private function loadCronClassesFromNamespace($namespace, $folder)
    {
        $cronClasses = [];
        
        $namespacePath = base_path(str_replace('\\', '/', $namespace));
        
        $files = File::allFiles($folder);
        
        
        foreach ($files as $file) {
            $className = $namespace . '\\' . pathinfo($file->getFilename(), PATHINFO_FILENAME);
            
            if (class_exists($className)) {
                $reflection = new ReflectionClass($className);
                
                if ($reflection->implementsInterface('App\Contracts\Cron') && !$reflection->isAbstract()) {
                    $cronClasses[] = $className;
                }
            }
        }
        

        return $cronClasses;
    }
}
