<?php

namespace App\Utils;

class FormBuilder
{
    protected $text = '<input type="text"';

    protected $textarea = '<textarea';

    protected $inputAttributes = [
        'class' => 'form-control',
    ];

    protected $wrapperAttributes = [];

    protected $html = '';

    function mergeAttributes($arrayOne, $arrayTwo)
    {
        $mergedArray = [];

        foreach ($arrayOne as $key => $value) {
            if (isset($arrayTwo[$key])) {
                $mergedValues = array_unique([$value, $arrayTwo[$key]]);
                $mergedArray[$key] = implode(' ', $mergedValues);
            } else {
                $mergedArray[$key] = $value;
            }
        }

        foreach ($arrayTwo as $key => $value) {
            if (!isset($mergedArray[$key])) {
                $mergedArray[$key] = $value;
            }
        }

        return $mergedArray;
    }

    function label($label)
    {
        $translatedLabel = htmlspecialchars(tt_trans($label, cons()::LANGUAGE_BACKEND), ENT_QUOTES, 'UTF-8');

        return '<label class="form-label">' . $translatedLabel . '</label>';
    }

    /**
     * Generate an HTML text input field with an optional wrapper <div>.
     *
     * @param string $name The name attribute for the input field.
     * @param string $label The label attribute for the input field.
     * @param string $value The value attribute for the input field.
     * @param string $type the value type is for which type want to produce.
     */

    // public function buildHtml(array $skeleton $type, $label, $name,$value = '')
    public function buildHtml(array $skeleton)
    {
        $html = '';
        foreach ($skeleton as $item) {
            $this->wrapperAttributes = [];

            $type = $item['type'];
            $label = $item['label'];
            $name = $item['name'];
            $value = isset($item['value']) ? $item['value'] : '';
            $size = isset($item['size']) ? $item['size'] : '100x100';
            $ai_enabled = isset($item['ai_enabled']) ? $item['ai_enabled'] : false;
            $inputCollection = isset($item['collection']) ? $item['collection'] : [];

            $this->wrapperAttributes = $this->mergeAttributes($this->wrapperAttributes, $item['parent_attributes']);

            if (!empty($this->wrapperAttributes)) {
                $html .= '<div';

                // Add any additional wrapper attributes
                foreach ($this->wrapperAttributes as $key => $attrValue) {
                    $html .= ' ' . htmlspecialchars($key, ENT_QUOTES, 'UTF-8') . '="' . htmlspecialchars($attrValue, ENT_QUOTES, 'UTF-8') . '"';
                }
                $html .= '>';
            }


            if (in_array($type, ['image', 'icon', 'select2'])) {
                if ($type == 'image') {
                    $html .= $this->label($label);
                    $html .= $this->$type($name, $value, $size);
                } else {
                    $html .= $this->label($label);
                    $html .= $this->$type($name, $value, $inputCollection);
                }
            } else {
                if ($ai_enabled) {
                    $view = view('components.ai', [
                        'label' =>$label,
                        'name' => htmlspecialchars($name, ENT_QUOTES, 'UTF-8'),
                        'target' => htmlspecialchars($name, ENT_QUOTES, 'UTF-8'),
                        'type' => $type,
                        'value' => $value,
                    ])->render();

                    $html .= $view;
                }else{

                    $html .= $this->label($label);
                    $html .= $this->$type;

                    // Add the name attribute
                    $html .= ' name="' . htmlspecialchars($name, ENT_QUOTES, 'UTF-8') . '"';

                    if ($type != 'textarea') {
                        if ($value !== '') {
                            $html .= ' value="' . $value . '"';
                        }
                    }

                    $this->inputAttributes = $this->mergeAttributes($this->inputAttributes, $item['input_attributes'] ?? []);

                    // Add any additional attributes
                    foreach ($this->inputAttributes as $key => $attrValue) {
                        $html .= ' ' . htmlspecialchars($key, ENT_QUOTES, 'UTF-8') . '="' . htmlspecialchars($attrValue, ENT_QUOTES, 'UTF-8') . '"';
                    }

                    if ($type == 'textarea') {
                        if ($value !== '') {
                            $html .= '>' . htmlspecialchars($value, ENT_QUOTES, 'UTF-8') . '</textarea>';
                        } else {
                            $html .= '></textarea>';
                        }
                    } else {
                        $html .= '>';
                    }
                }

            }

            // Close the <div> wrapper if it was opened
            if (!empty($this->wrapperAttributes)) {
                $html .= '</div>';
            }
        }

        return $html;
    }

    function image($name, $value, $size)
    {
        return '<div class="drop-zone drop-zone-sm border border-2 border-dashed border-light-primary rounded p-2 d-flex flex-column align-items-center justify-content-center bg-light-primary overflow-hidden text-center">
                <div class="prompt-text h-100 w-100 d-flex flex-column align-items-center justify-content-center">
                    <span class="display-6 text-primary mb-3">
                        <iconify-icon icon="solar:gallery-linear"></iconify-icon>
                    </span>
                    <h6 class="mb-1 text-sm"><span class="font-semibold">Click to upload</span> or drag and drop</h6>
                    <p class="text-xs mb-0 text-muted">SVG, PNG, JPG or GIF (MAX. 800x400px)</p>
                </div>
                <input name="' .
            $name .
            '"  type="file" class="opacity-0 visually-hidden drop-zone-input"/>

            <div class="drop-zone-thumb" style="background-image:url(' .
            getFile('cms', $value, $size) .
            ')">
                                            <div class="overlay"><p class="thumbnail-file-name"></p><p class="overlay-prompt-text">' .
            tt_trans('Drag and drop or click to replace', cons()::LANGUAGE_BACKEND) .
            '</p></div>
                                        </div>
            </div>';
    }

    function icon($name, $value)
    {
        return '<div class="input-group">
                        <input name="' .
            $name .
            '" data-placement="bottomRight" class="form-control iconPicker" value="' .
            htmlspecialchars($value, ENT_QUOTES, 'UTF-8', false) .
            '"
                        type="text"/>
                        <span class="input-group-text input-group-addon dropdown-toggle"></span>
                    </div> <p class="text-xs">Supported icons - <a href="https://fontawesome.com/icons" target="_blank" class="text-primary">Fontawesome</a> | <a href="https://icons.getbootstrap.com/" target="_blank" class="text-primary">BootStrap</a> | <a href="https://remixicon.com/" target="_blank" class="text-primary">Remix</a> | <a href="https://icon-sets.iconify.design/" target="_blank" class="text-primary">Iconify</a></p>';
    }

    function select2($name, $value, $inputCollection)
    {
        $option = '';

        foreach ($inputCollection as $col) {
            $is_selected = '';

            if (in_array($col->id, $value)) {
                $is_selected = ' selected';
            }
            $option .= '<option value=' . $col->id . '' . $is_selected . '>' . $col->name . '</option>';
        }
        return '<select class="form-select-two" name="' .
            $name .
            '" multiple="multiple">
                       ' .
            $option .
            '
                    </select>';
    }
}
